// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/system/bluetooth/bluetooth_detailed_view_impl.h"

#include <memory>

#include "ash/public/cpp/test/test_system_tray_client.h"
#include "ash/system/bluetooth/bluetooth_device_list_item_view.h"
#include "ash/system/tray/detailed_view_delegate.h"
#include "ash/test/ash_test_base.h"
#include "mojo/public/cpp/bindings/clone_traits.h"
#include "ui/views/test/views_test_utils.h"
#include "ui/views/widget/widget.h"

namespace ash {

namespace {

using bluetooth_config::mojom::BluetoothDeviceProperties;
using bluetooth_config::mojom::PairedBluetoothDeviceProperties;
using bluetooth_config::mojom::PairedBluetoothDevicePropertiesPtr;

const char kDeviceId[] = "/device/id";

class FakeBluetoothDetailedViewDelegate
    : public BluetoothDetailedView::Delegate {
 public:
  FakeBluetoothDetailedViewDelegate() = default;
  ~FakeBluetoothDetailedViewDelegate() override = default;

  // BluetoothDetailedView::Delegate:
  void OnToggleClicked(bool new_state) override {}

  void OnPairNewDeviceRequested() override {
    ++pair_new_device_requested_count_;
  }

  void OnDeviceListItemSelected(
      const PairedBluetoothDevicePropertiesPtr& device) override {
    last_device_list_item_selected_ = mojo::Clone(device);
  }

  int pair_new_device_requested_count_ = 0;
  PairedBluetoothDevicePropertiesPtr last_device_list_item_selected_;
};

// This class exists to stub out the CloseBubble() call. This allows tests to
// directly construct the detailed view, without depending on the entire quick
// settings bubble and view hierarchy.
class FakeDetailedViewDelegate : public DetailedViewDelegate {
 public:
  FakeDetailedViewDelegate()
      : DetailedViewDelegate(/*tray_controller=*/nullptr) {}
  ~FakeDetailedViewDelegate() override = default;

  // DetailedViewDelegate:
  void CloseBubble() override { ++close_bubble_count_; }

  int close_bubble_count_ = 0;
};

}  // namespace

class BluetoothDetailedViewImplTest : public AshTestBase {
 public:
  void SetUp() override {
    AshTestBase::SetUp();

    auto bluetooth_detailed_view = std::make_unique<BluetoothDetailedViewImpl>(
        &detailed_view_delegate_, &bluetooth_detailed_view_delegate_);
    bluetooth_detailed_view_ = bluetooth_detailed_view.get();

    widget_ = CreateFramelessTestWidget();
    widget_->SetFullscreen(true);
    widget_->SetContentsView(bluetooth_detailed_view.release()->GetAsView());
  }

  void TearDown() override {
    widget_.reset();

    AshTestBase::TearDown();
  }

  views::Button* GetSettingsButton() {
    return bluetooth_detailed_view_->settings_button_;
  }

  views::Button* GetPairNewDeviceView() {
    return bluetooth_detailed_view_->pair_new_device_view_;
  }

  std::unique_ptr<views::Widget> widget_;
  FakeBluetoothDetailedViewDelegate bluetooth_detailed_view_delegate_;
  FakeDetailedViewDelegate detailed_view_delegate_;
  BluetoothDetailedViewImpl* bluetooth_detailed_view_ = nullptr;
};

TEST_F(BluetoothDetailedViewImplTest, PressingSettingsButtonOpensSettings) {
  views::Button* settings_button = GetSettingsButton();

  // Clicking the button at the lock screen does nothing.
  GetSessionControllerClient()->SetSessionState(
      session_manager::SessionState::LOCKED);
  LeftClickOn(settings_button);
  EXPECT_EQ(0, GetSystemTrayClient()->show_bluetooth_settings_count());
  EXPECT_EQ(0, detailed_view_delegate_.close_bubble_count_);

  // Clicking the button in an active user session opens OS settings.
  GetSessionControllerClient()->SetSessionState(
      session_manager::SessionState::ACTIVE);
  LeftClickOn(settings_button);
  EXPECT_EQ(1, GetSystemTrayClient()->show_bluetooth_settings_count());
  EXPECT_EQ(1, detailed_view_delegate_.close_bubble_count_);
}

TEST_F(BluetoothDetailedViewImplTest, PressingPairNewDeviceNotifiesDelegate) {
  bluetooth_detailed_view_->UpdateBluetoothEnabledState(true);
  views::test::RunScheduledLayout(bluetooth_detailed_view_);

  // Clicking the "pair new device" row notifies the delegate.
  views::Button* pair_new_device_view = GetPairNewDeviceView();
  LeftClickOn(pair_new_device_view);
  EXPECT_EQ(1,
            bluetooth_detailed_view_delegate_.pair_new_device_requested_count_);
}

TEST_F(BluetoothDetailedViewImplTest, SelectingDeviceListItemNotifiesDelegate) {
  bluetooth_detailed_view_->UpdateBluetoothEnabledState(true);

  // Create a simulated device and add it to the list.
  PairedBluetoothDevicePropertiesPtr paired_properties =
      PairedBluetoothDeviceProperties::New();
  paired_properties->device_properties = BluetoothDeviceProperties::New();
  paired_properties->device_properties->id = kDeviceId;

  BluetoothDeviceListItemView* device_list_item =
      bluetooth_detailed_view_->AddDeviceListItem();
  device_list_item->UpdateDeviceProperties(
      /*device_index=*/0, /*total_device_count=*/0, paired_properties);

  bluetooth_detailed_view_->NotifyDeviceListChanged();

  // Clicking on the item notifies the delegate that the device was selected.
  LeftClickOn(device_list_item);
  EXPECT_TRUE(
      bluetooth_detailed_view_delegate_.last_device_list_item_selected_);
  EXPECT_EQ(kDeviceId,
            bluetooth_detailed_view_delegate_.last_device_list_item_selected_
                ->device_properties->id);
}

}  // namespace ash
