// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "internal/platform/implementation/windows/test_utils.h"

#include <shlobj.h>

#include <algorithm>
#include <sstream>

#include "absl/strings/str_format.h"
#include "absl/strings/str_replace.h"

namespace test_utils {
std::wstring StringToWideString(const std::string& s) {
  int len;
  int slength = (int)s.length() + 1;
  len = MultiByteToWideChar(CP_ACP, 0, s.c_str(), slength, 0, 0);
  wchar_t* buf = new wchar_t[len];
  MultiByteToWideChar(CP_ACP, 0, s.c_str(), slength, buf, len);
  std::wstring r(buf);
  delete[] buf;
  return r;
}

std::string GetPayloadPath(location::nearby::PayloadId payload_id) {
  PWSTR basePath;

  // Retrieves the full path of a known folder identified by the folder's
  // KNOWNFOLDERID.
  // https://docs.microsoft.com/en-us/windows/win32/api/shlobj_core/nf-shlobj_core-shgetknownfolderpath
  SHGetKnownFolderPath(
      FOLDERID_Downloads,  //  rfid: A reference to the KNOWNFOLDERID that
                           //  identifies the folder.
      0,           // dwFlags: Flags that specify special retrieval options.
      NULL,        // hToken: An access token that represents a particular user.
      &basePath);  // ppszPath: When this method returns, contains the address
                   // of a pointer to a null-terminated Unicode string that
                   // specifies the path of the known folder. The calling
                   // process is responsible for freeing this resource once it
                   // is no longer needed by calling CoTaskMemFree, whether
                   // SHGetKnownFolderPath succeeds or not.

  size_t bufferSize;
  // Get the required buffer size.
  wcstombs_s(&bufferSize, NULL, 0, basePath, 0);
  std::string fullpathUTF8(bufferSize, NULL);
  wcstombs_s(&bufferSize, fullpathUTF8.data(), bufferSize, basePath, _TRUNCATE);
  std::string fullPath = std::string(fullpathUTF8);
  // Clean up the string by removing null's
  fullPath.erase(std::find(fullPath.begin(), fullPath.end(), '\0'),
                 fullPath.end());

  CoTaskMemFree(basePath);

  std::string path =
      absl::StrFormat("%s\\%s", fullPath, std::to_string(payload_id));

  path = absl::StrReplaceAll(path, {{"\\", "/"}});

  return path;
}

}  // namespace test_utils
