// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#ifndef THIRD_PARTY_NEARBY_PRESENCE_IMPLEMENTATION_SCAN_MANAGER_H_
#define THIRD_PARTY_NEARBY_PRESENCE_IMPLEMENTATION_SCAN_MANAGER_H_

#include <cstdint>
#include <memory>
#include <string>
#include <utility>

#include "absl/container/flat_hash_map.h"
#include "internal/platform/single_thread_executor.h"
#include "internal/proto/credential.pb.h"
#include "presence/data_types.h"
#include "presence/implementation/advertisement_decoder.h"
#include "presence/implementation/credential_manager.h"
#include "presence/implementation/mediums/mediums.h"
#include "presence/scan_request.h"

namespace nearby {
namespace presence {

// The instance of ScanManager is owned by `ServiceControllerImpl`.
// Helping service controller to manage scan requests and callbacks.
class ScanManager {
 public:
  using SingleThreadExecutor = ::location::nearby::SingleThreadExecutor;
  using Mutex = ::location::nearby::Mutex;
  using MutexLock = ::location::nearby::MutexLock;
  using ScanningSession =
      ::location::nearby::api::ble_v2::BleMedium::ScanningSession;
  using Runnable = ::location::nearby::Runnable;
  using BleAdvertisementData =
      ::location::nearby::api::ble_v2::BleAdvertisementData;
  using PublicCredential = ::nearby::internal::PublicCredential;
  using IdentityType = ::nearby::internal::IdentityType;

  ScanManager(Mediums& mediums, CredentialManager& credential_manager,
              SingleThreadExecutor& executor) {
    mediums_ = &mediums, credential_manager_ = &credential_manager;
    executor_ = &executor;
  }
  ~ScanManager() = default;

  ScanSessionId StartScan(ScanRequest scan_request, ScanCallback cb);
  void StopScan(ScanSessionId session_id);
  // Below functions are test only.
  // Reference: go/totw/135#augmenting-the-public-api-for-tests
  int ScanningCallbacksLengthForTest();

 private:
  struct ScanSessionState {
    ScanRequest request;
    ScanCallback callback;
    absl::flat_hash_map<IdentityType, std::vector<PublicCredential>>
        credentials;
    AdvertisementDecoder decoder;
    std::unique_ptr<ScanningSession> scanning_session;
  };
  void NotifyFoundBle(ScanSessionId id, BleAdvertisementData data,
                      absl::string_view remote_address)
      ABSL_EXCLUSIVE_LOCKS_REQUIRED(*executor_);
  void FetchCredentials(ScanSessionId id, const ScanRequest& scan_request)
      ABSL_EXCLUSIVE_LOCKS_REQUIRED(*executor_);
  void UpdateCredentials(ScanSessionId id, IdentityType identity_type,
                         std::vector<PublicCredential> credentials)
      ABSL_EXCLUSIVE_LOCKS_REQUIRED(*executor_);
  void RunOnServiceControllerThread(absl::string_view name, Runnable runnable) {
    executor_->Execute(std::string(name), std::move(runnable));
  }
  Mediums* mediums_;
  CredentialManager* credential_manager_;
  absl::flat_hash_map<ScanSessionId, ScanSessionState> scan_sessions_
      ABSL_GUARDED_BY(*executor_);
  SingleThreadExecutor* executor_;
};

}  // namespace presence
}  // namespace nearby

#endif  // THIRD_PARTY_NEARBY_PRESENCE_IMPLEMENTATION_SCAN_MANAGER_H_
