/* Any copyright is dedicated to the Public Domain.
   http://creativecommons.org/publicdomain/zero/1.0/ */
/* import-globals-from helper_html_tooltip.js */

"use strict";

/**
 * Test the HTMLTooltip "doorhanger" type's hang direction. It should hang
 * along the flow of text e.g. in RTL mode it should hang left and in LTR mode
 * it should hang right.
 */

const HTML_NS = "http://www.w3.org/1999/xhtml";
const TEST_URI = CHROME_URL_ROOT + "doc_html_tooltip_doorhanger-01.xhtml";

const {
  HTMLTooltip,
} = require("resource://devtools/client/shared/widgets/tooltip/HTMLTooltip.js");
loadHelperScript("helper_html_tooltip.js");

let useXulWrapper;

add_task(async function () {
  // Force the toolbox to be 200px high;
  await pushPref("devtools.toolbox.footer.height", 200);

  await addTab("about:blank");
  const { doc } = await createHost("bottom", TEST_URI);

  info("Run tests for a Tooltip without using a XUL panel");
  useXulWrapper = false;
  await runTests(doc);
});

async function runTests(doc) {
  info("Create HTML tooltip");
  const tooltip = new HTMLTooltip(doc, { type: "doorhanger", useXulWrapper });
  const div = doc.createElementNS(HTML_NS, "div");
  div.style.width = "200px";
  div.style.height = "35px";
  tooltip.panel.appendChild(div);

  const anchors = [...doc.querySelectorAll(".anchor")];
  for (const anchor of anchors) {
    const hangDirection = anchor.getAttribute("data-hang");

    info("Display the tooltip on an anchor.");
    await showTooltip(tooltip, anchor);

    const arrow = tooltip.arrow;
    ok(arrow, "Tooltip has an arrow");

    // Get the geometry of the the tooltip panel & arrow.
    const panelBounds = tooltip.panel
      .getBoxQuads({ relativeTo: doc })[0]
      .getBounds();
    const arrowBounds = arrow.getBoxQuads({ relativeTo: doc })[0].getBounds();
    const panelBoundsCentre = (panelBounds.left + panelBounds.right) / 2;
    const arrowCentre = (arrowBounds.left + arrowBounds.right) / 2;

    if (hangDirection === "left") {
      ok(
        arrowCentre > panelBoundsCentre,
        `tooltip hangs to the left for ${anchor.id}`
      );
    } else {
      ok(
        arrowCentre < panelBoundsCentre,
        `tooltip hangs to the right for ${anchor.id}`
      );
    }

    await hideTooltip(tooltip);
  }

  tooltip.destroy();
}
